###############################
# READ AND PROCESS INPUT DATA #
##########################################################################################

# STATPOP. -------------------------------------------------------------------------------

POP <- 
  read_delim(PAR$in_pop) %>% 
  # Filter for 'ständige Wohnbevölkerung der Schweiz Ende Dezember'.
  filter(jahr %in% 2001:first(PAR$pint), variable == "bevendejahr") %>% 
  select(sex, nat, age = alt, year = jahr, pop = value)

# Pension counts by year, sex, nationality, pension type and age. ------------------------

PEN_N <-
  read_delim(PAR$in_rr) %>%
  filter(tsex    != "Sexe - total"       ,
         twohn   == "Suisse"             ,
         tnation != "Nationalité - total",
         lage    != - 99999) %>%
  select(year   = an_rr, sex = tsex, nat = tnation, age = lage,
         # Prefix 'n' refers to pension counts.
         n_alt  = RAV ,
         n_wit  = RAS ,
         n_waiv = RAOp,
         n_waim = RAOm,
         n_waid = RAOd,
         n_kinv = RAEp,
         n_kinm = RAEm) %>%
  mutate(sex = recode(sex, "Homme"    = "m" , "Femme"  = "f"),
         nat = recode(nat, "Etranger" = "au", "Suisse" = "ch"),
         # Consolidate age ranges for consistency with STATPOP.
         age = pmin(99, age)) %>%
  pivot_longer(cols = contains("n_"), names_to = "type", names_prefix = "n_",
               values_to = "n") %>%
  # Dispense with father-mother distinctions in pension types and treat 'doppelte 
  # Waisenrenten' like simple ones to avoid overly sparse time series.
  mutate(type = fct_collapse(type,
                             wai = c("waiv", "waim", "waid"),
                             kin = c("kinv", "kinm"))) %>% 
  # Consolidate groups regarding age 99+ and discarded pension type distinctions.
  summarize(n = sum(n), .by = c("sex", "nat", "type", "age", "year")) %>% 
  # Make implicit missing values explicit.
  complete(sex, nat, type, age, year, fill = list(n = 0, m = 0))

# Pension counts and levels by year, sex, domicile and pension type. ---------------------

PEN_NM <-
  read_delim(PAR$in_rr) %>%
  filter(tsex    != "Sexe - total"             ,
         twohn   != "Pays de résidence - total",
         tnation == "Nationalité - total"      ,
         lage    != - 99999) %>%
  select(year = an_rr, sex = tsex, dom = twohn, age = lage,
         # Prefix 'n' refers to pension counts and 'm' to the mean pension level per 
         # recipient relative to the legal minimum pension of the respective year.
         n_alt  = RAV , m_alt  = RAV_moy ,
         n_wit  = RAS , m_wit  = RAS_moy ,
         n_waiv = RAOp, m_waiv = RAOp_moy,
         n_waim = RAOm, m_waim = RAOm_moy,
         n_waid = RAOd, m_waid = RAOd_moy,
         n_kinv = RAEp, m_kinv = RAEp_moy,
         n_kinm = RAEm, m_kinm = RAEm_moy) %>%
  pivot_longer(cols = contains(c("n_", "m_")), names_to = c("metric", "type"),
               names_sep = "_") %>%
  mutate(sex = recode(sex, "Homme"    = "m" , "Femme"  = "f") ,
         dom = recode(dom, "Etranger" = "au", "Suisse" = "ch")) %>%
  pivot_wider(names_from = metric, values_from = value, values_fn = sum) %>%
  select(year, sex, dom, type, n, m) %>%
  # Dispense with father-mother distinctions in pension types and treat 'doppelte 
  # Waisenrenten' like simple ones to avoid overly sparse time series.
  mutate(type = fct_collapse(type,
                             wai = c("waiv", "waim", "waid"),
                             kin = c("kinv", "kinm"))) %>% 
  # Consolidate groups regarding the discarded pension type distinctions. 
  summarize(m = weighted.mean(m, n), n = sum(n), 
            .by = c("sex", "dom", "type", "year"))

# BFS population scenario. ---------------------------------------------------------------

SCEN <-
  read_delim(PAR$in_scen) %>% 
  select(scen = scenario, year = jahr, sex, nat, age = alt, val = bevendejahr) %>%
  # Filter for relevant editions of the demographic scenarios.
  mutate(scen = 
           fct_recode(scen, 
                      "A" = paste0("A_00_", PAR$scen),
                      "B" = paste0("B_00_", PAR$scen),
                      "C" = paste0("C_00_", PAR$scen))
         ) %>%
  filter(scen %in% c("A", "B", "C")) %>%
  mutate(scen = fct_drop(scen), 
         # Consolidate higher age ranges for consistency with STATPOP.
         age = pmin(99, age)) %>% 
  summarize(val = sum(val), 
                   .by = c("scen", "sex", "nat", "age", "year")) %>% 
  # Only keep latest observed year for adjustment purposes. For the publication in 2025,
  # the year 2024 data represents provisional STATPOP numbers provided by the BFS.
  filter(year %in% PAR$pint) 

# Legal minimal pensions, inflation rates and AHV fund returns. --------------------------

# Historic legal minimal pensions as well as projected ones based on given SLI/LIK
# projections.
MINIMAL_PENSION <-
  read_delim(PAR$in_mp) %>% 
  select(year = jahr, mp = minimalrente) %>%
  # Upscale to yearly numbers.
  mutate(mp = 12 * mp)

INFLATION <-
  read_delim(PAR$in_inf) %>% 
  select(year = jahr, inf = lik_basis_1977)

ECKWERTE <-
  read_delim(PAR$in_eck) %>%
  select(year = jahr, df = preis) %>% 
  complete(year = first(PAR$pint):last(PAR$pint)) %>% 
  fill(df) %>% 
  # Calculate discount factors for 'real' Finanzhaushalte.
  mutate(df = ifelse(year == first(PAR$pint), 0, df),
         df = cumprod(1 / (1 + df / 100)))

# Widow projections from complementary model. --------------------------------------------

WIDOWS <- 
  read_delim(PAR$in_wid) %>% 
  mutate(type = "wit",
         scen = 
           fct_recode(scen, 
                      "A" = paste0("A_00_", PAR$scen),
                      "B" = paste0("B_00_", PAR$scen),
                      "C" = paste0("C_00_", PAR$scen))
         ) %>% 
  select(scen, dom, sex, type, year = jahr, m, n) %>% 
  left_join(MINIMAL_PENSION, by = "year")

# Yearly ZAS expenditure data. -----------------------------------------------------------

ZAS <-
  read_delim(PAR$in_zas)

# Cost effect of reform AHV 21 excluding savings due to reference age increase for women.-

AHV21_COST <- 
  read_delim(PAR$in_ahv21) %>% 
  mutate(cost = cost * 1e6)
